import { EventProcessor, Hub, Integration } from '@sentry/types';
import { Debugger, InspectorNotification } from 'inspector';
type Variables = Record<string, unknown>;
type OnPauseEvent = InspectorNotification<Debugger.PausedEventDataType>;
export interface DebugSession {
    /** Configures and connects to the debug session */
    configureAndConnect(onPause: (message: OnPauseEvent, complete: () => void) => void, captureAll: boolean): void;
    /** Updates which kind of exceptions to capture */
    setPauseOnExceptions(captureAll: boolean): void;
    /** Gets local variables for an objectId */
    getLocalVariables(objectId: string, callback: (vars: Variables) => void): void;
}
type Next<T> = (result: T) => void;
type Add<T> = (fn: Next<T>) => void;
type CallbackWrapper<T> = {
    add: Add<T>;
    next: Next<T>;
};
type RateLimitIncrement = () => void;
/**
 * Creates a rate limiter
 * @param maxPerSecond Maximum number of calls per second
 * @param enable Callback to enable capture
 * @param disable Callback to disable capture
 * @returns A function to call to increment the rate limiter count
 */
export declare function createRateLimiter(maxPerSecond: number, enable: () => void, disable: (seconds: number) => void): RateLimitIncrement;
/** Creates a container for callbacks to be called sequentially */
export declare function createCallbackList<T>(complete: Next<T>): CallbackWrapper<T>;
export interface FrameVariables {
    function: string;
    vars?: Variables;
}
interface Options {
    /**
     * Capture local variables for both caught and uncaught exceptions
     *
     * - When false, only uncaught exceptions will have local variables
     * - When true, both caught and uncaught exceptions will have local variables.
     *
     * Defaults to `true`.
     *
     * Capturing local variables for all exceptions can be expensive since the debugger pauses for every throw to collect
     * local variables.
     *
     * To reduce the likelihood of this feature impacting app performance or throughput, this feature is rate-limited.
     * Once the rate limit is reached, local variables will only be captured for uncaught exceptions until a timeout has
     * been reached.
     */
    captureAllExceptions?: boolean;
    /**
     * Maximum number of exceptions to capture local variables for per second before rate limiting is triggered.
     */
    maxExceptionsPerSecond?: number;
}
/**
 * Adds local variables to exception frames
 *
 * Default: 50
 */
export declare class LocalVariables implements Integration {
    private readonly _options;
    private readonly _session;
    static id: string;
    readonly name: string;
    private readonly _cachedFrames;
    private _rateLimiter;
    constructor(_options?: Options, _session?: DebugSession | undefined);
    /**
     * @inheritDoc
     */
    setupOnce(addGlobalEventProcessor: (callback: EventProcessor) => void, getCurrentHub: () => Hub): void;
    /** Setup in a way that's easier to call from tests */
    private _setup;
    /**
     * Handle the pause event
     */
    private _handlePaused;
    /**
     * Adds local variables event stack frames.
     */
    private _addLocalVariables;
    /**
     * Adds local variables to the exception stack frames.
     */
    private _addLocalVariablesToException;
}
export {};
//# sourceMappingURL=localvariables.d.ts.map
